// -*- C++ -*-
#ifndef __INTERPOLATOR_H__
#define __INTERPOLATOR_H__

#include <string>
#include <iostream>

template<typename V, typename T> class InterpolatorMinjerk;
template<typename V, typename T> class InterpolatorLinear;

template<typename V, typename T = double>
class Interpolator  // interface
{
public:
    static Interpolator* create(std::string name) {
        if (name == "minjerk") return new InterpolatorMinjerk<V,T>();
        if (name == "linear") return new InterpolatorLinear<V,T>();
        return NULL;
    }
    Interpolator() 
        : m_xi(0),
          m_vi(0),
          m_ai(0),
          m_xf(0),
          m_vf(0),
          m_af(0),
          m_time(0)
        {
        }
    Interpolator(T time, V xi, V vi, V ai, V xf, V vf, V af)
        : m_xi(xi),
          m_vi(vi),
          m_ai(ai),
          m_xf(xf),
          m_vf(vf),
          m_af(af),
          m_time(time)
        {
            init();
        }
    T getTime() {return m_time;}
    V getInitialPos() {return m_xi;}
    V getInitialVel() {return m_vi;}
    V getInitialAccel() {return m_ai;}
    V getFinalPos() {return m_xf;}
    V getFinalVel() {return m_vf;}
    V getFinalAccel() {return m_af;}
    void setBoundaryCondition(T time, V xi, V vi, V ai, V xf, V vf, V af)
        {
            m_xi = xi;
            m_vi = vi;
            m_ai = ai;
            m_xf = xf;
            m_vf = vf;
            m_af = af;
            m_time = time;
            init();
        }
    void setBoundaryCondition(T time, V xf, V vf, V af)
        {
            m_xf = xf;
            m_vf = vf;
            m_af = af;
            m_time = time;
            init();
        }
    virtual V getPos(T t) = 0;
    virtual V getVel(T t) = 0;
    virtual V getAccel(T t) = 0;
    virtual ~Interpolator() {}
protected:
    virtual void init() = 0;
    V m_xi;
    V m_vi;
    V m_ai;
    V m_xf;
    V m_vf;
    V m_af;
    T m_time;
private:
};

template<typename V, typename T = double>
class InterpolatorMinjerk 
    : public Interpolator<V, T>
{
public:
    InterpolatorMinjerk() :a0(0),a1(0),a2(0),a3(0),a4(0),a5(0),A(0),B(0),C(0) {};
    virtual V getPos(T t) {return a0+a1*t+a2*t*t+a3*t*t*t+a4*t*t*t*t+a5*t*t*t*t*t;}
    virtual V getVel(T t) {return a1+2*a2*t+3*a3*t*t+4*a4*t*t*t+5*a5*t*t*t*t;}
    virtual V getAccel(T t) {return 2*a2+6*a3*t+12*a4*t*t+20*a5*t*t*t;}
protected:
    virtual void init() {
        A=(this->m_xf-(this->m_xi+this->m_vi*this->m_time+(this->m_ai/2)*this->m_time*this->m_time))/(this->m_time*this->m_time*this->m_time);
        B=(this->m_vf-(this->m_vi+this->m_ai*this->m_time))/(this->m_time*this->m_time);
        C=(this->m_af-this->m_ai)/this->m_time;
        a0=this->m_xi;
        a1=this->m_vi;
        a2=this->m_ai/2;
        a3=10*A-4*B+C/2;
        a4=(-15*A+7*B-C)/this->m_time;
        a5=(6*A-3*B+C/2)/(this->m_time*this->m_time);
    }
private:
    V a0,a1,a2,a3,a4,a5;
    V A,B,C;
};

template<typename V, typename T = double>
class InterpolatorLinear
    : public Interpolator <V, T>
{
public:
    InterpolatorLinear() : slope(0){}
    virtual V getPos(T t) {return slope*t + this->m_xi;}
    virtual V getVel(T t) {return slope;}
    virtual V getAccel(T t) {return 0;}

protected:
    virtual void init() {
        slope = (this->m_xf - this->m_xi)/this->m_time;
    }
private:
    V slope;
};
#endif //__INTERPOLATOR_H__
