// -*- mode: C++; coding: utf-8-unix; -*-
#include "Seq.h"

// Module specification
// <rtc-template block="module_spec">
static const char* seq_spec[] =
{
    "implementation_id", "Seq",
    "type_name",         "Seq",
    "description",       "Seq component",
    "version",           "1.0",
    "vendor",            "JSK",
    "category",          "example",
    "activity_type",     "DataFlowComponent",
    "max_instance",      "10",
    "language",          "C++",
    "lang_type",         "compile",
    // Configuration variables
    "conf.default.componentconf_file",         "SeqTest.conf",
    "conf.default.config", "Seq0",
    ""
};
// </rtc-template>

Seq::Seq(RTC::Manager* manager)
    : RTC::DataFlowComponentBase(manager),
      // <rtc-template block="initializer">
      m_inStatePort("input", m_inState),
      m_outStatePort("output", m_outState),
      m_stateServicePort("StateService"),
      m_CommandServicePort("CommandService")
      // </rtc-template>
{
}

Seq::~Seq()
{
}

// void Seq::init()
// {

// }

RTC::ReturnCode_t Seq::onInitialize()
{
    // Registration: InPort/OutPort/Service
    // <rtc-template block="registration">
    // Set InPort buffers
    addInPort("input", m_inStatePort);

    // Set OutPort buffer
    addOutPort("output", m_outStatePort);
    std::string portname = std::string(getInstanceName()) + std::string(".") + std::string("output");
    m_outStateManager.registerStateServiceToOutPort(m_portAdmin.getPort(portname.c_str()));
    
    // Set service provider to Ports
    m_stateServicePort.registerProvider("stateservice", "StateService", m_stateServiceManager);
    m_CommandServicePort.registerProvider("commandservice", "CommandService", m_commandservice);

    // Set service consumers to Ports

    // Set CORBA Service Ports
    addPort(m_stateServicePort);
    addPort(m_CommandServicePort);

    // </rtc-template>
    m_commandservice.registerCommand("angle-vector", "help for angle-vector",
                                   boost::bind( &Seq::angle_vector, this, _1 ));

    m_commandservice.registerCommand("wait-interpolation", "help for wait-interpolation",
                                   boost::bind( &Seq::wait_interpolation, this, _1 ));

    m_commandservice.registerCommand("wait-all-interpolation", "help for wait-interpolation",
                                   boost::bind( &Seq::wait_all_interpolation, this, _1 ));

    m_commandservice.registerCommand("adapt", "help for adapt",
                                   boost::bind( &Seq::adapt, this, _1 ));

    m_commandservice.registerCommand("adapt-1motor", "help for adapt-1motor",
                                   boost::bind( &Seq::adapt_1motor, this, _1 ));

    m_commandservice.registerCommand("adapt-allseq", "help for adapt-allseq",
                                   boost::bind( &Seq::adapt_allseq, this, _1 ));

    m_commandservice.registerCommand("adapt-1motor-allseq", "help for adapt-1motor-allseq",
                                   boost::bind( &Seq::adapt_1motor_allseq, this, _1 ));

    m_commandservice.registerCommand("angle-vector-sequence", "help for angle-vector-sequence",
                                   boost::bind( &Seq::angle_vector_sequence, this, _1 ));

    m_commandservice.registerCommand("set-anglevectortimelist", "help for set-anglevectortimelist",
                                   boost::bind( &Seq::set_anglevectortimelist, this, _1 ));

    m_commandservice.registerCommand("create-newsequencer", "help for create-newsequencer",
                                   boost::bind( &Seq::create_newsequencer, this, _1 ));

    m_commandservice.registerCommand("set-sequencer-priority", "help for set-sequencer-priority",
                                   boost::bind( &Seq::set_sequencer_priority, this, _1 ));

    m_commandservice.registerCommand("get-sequencer-priority", "help for get-sequencer-priority",
                                   boost::bind( &Seq::get_sequencer_priority, this, _1 ));

    m_commandservice.registerCommand("set-sequencer-mask", "help for set-sequencer-mask",
                                   boost::bind( &Seq::set_sequencer_mask, this, _1 ));

    m_commandservice.registerCommand("get-sequencer-mask", "help for get-sequencer-mask",
                                   boost::bind( &Seq::get_sequencer_mask, this, _1 ));

    m_commandservice.registerCommand("set-sequencer-integrationmethod", "help for set-sequencer-integrationmethod",
                                   boost::bind( &Seq::set_sequencer_integrationmethod, this, _1 ));

    m_commandservice.registerCommand("set-sequencer-motionspeed", "help for set-sequencer-motionspeed",
                                   boost::bind( &Seq::set_sequencer_motionspeed, this, _1 ));
    
    m_commandservice.registerCommand("set-sequencer-motiongain", "help for set-sequencer-motiongain",
                                   boost::bind( &Seq::set_sequencer_motiongain, this, _1 ));

    m_anglevel_limit = 100.0;//not degree, pulse

    // Bind variables and configuration variable
    bindParameter("componentconf_file", m_componentconf_file, "SeqTest.conf");
    bindParameter("config", m_config, "Seq0");
    
    return RTC::RTC_OK;
}


/*
  RTC::ReturnCode_t Seq::onFinalize()
  {
  return RTC::RTC_OK;
  }
*/

/*
  RTC::ReturnCode_t Seq::onStartup(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/

/*
  RTC::ReturnCode_t Seq::onShutdown(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/


RTC::ReturnCode_t Seq::onActivated(RTC::UniqueId ec_id)
{
    // 設定ファイルの読み込み
    coil::Properties fprop;    
    std::ifstream compf(m_componentconf_file.c_str());
    if (compf.fail()) 
    {
        std::cerr << "failed to open component configuration file: "<< m_componentconf_file << std::endl;
        return RTC::RTC_ERROR;
    }
    fprop.load(compf);

    // ステート作成開始
    // 全部クリア
    m_seqStateManager.clear();
    m_seqInfos.clear();

    // 設定ファイルから自分に関係する部分を探す
    // なければエラー
    if (fprop.findNode(m_config)==0)
    {
        std::cerr << "faild to find config node" << std::endl;
        return RTC::RTC_ERROR;
    }
    
    coil::Properties prop;
    prop << fprop.getNode(m_config);
    std::vector<coil::Properties *> props;

    // SequencerInfoの追加
    std::vector<coil::Properties*> seqs = prop.getLeaf();
    for(unsigned int i = 0; i < seqs.size(); i++)
    {
        std::vector<RTC::Properties *> props = seqs[i]->getLeaf();
        if (props.size() > 0) {
            try {
                RTMEUS::StateProfile profile;
                // StateProfileをつくるために一度ステートを作る
                (*props[0])["type"] = "CHARSEQ"; // 型は適当
                StatePtr state = StateBase::create(*props[0]);
                // シーケンサ情報を登録
                SequencerInfo info;
                info.potentio_name = (*props[0])["in"]; // 入力と
                info.prof =  state->getStateProfile(); // data
                m_seqInfos[seqs[i]->getName()] = info;
                std::cerr << "Seq: " << seqs[i]->getName() << "[" << info.potentio_name << "]" << std::endl;
            }
            catch(std::runtime_error &e)
            {
                std::cerr << e.what() << std::endl;
                return RTC::RTC_ERROR;
            }
        }
    }

    std::cerr << std::setw(40) << std::setfill('=') << " " << std::endl;

    m_first = true;
    //
    m_sequencers.clear();
    return RTC::RTC_OK;
}

RTC::ReturnCode_t Seq::onDeactivated(RTC::UniqueId ec_id)
{
    m_sequencers.clear();
    return RTC::RTC_OK;
}


RTC::ReturnCode_t Seq::onExecute(RTC::UniqueId ec_id)
{
    // 読み込み
    m_inStatePort.read();
    if (m_first || m_inState.dataset.profile_updated)
    {
        std::string portname = std::string(getInstanceName()) + std::string(".") + std::string("input");
        RTMEUS::StateService_var ss = m_inStateManager.getStateServiceFromInPort(m_portAdmin.getPort(portname.c_str()));
        if (!CORBA::is_nil(ss))
        {
            try
            {
                // まず管理するステートをまず全部クリア
                m_inStateManager.clear();
                m_outStateManager.clear();
                m_seqStateManager.clear();
                // 入力されたステートをinStateManagerにいれる
                RTMEUS::StateProfileList_var profs = ss->get_state_profiles( );
                m_inStateManager.addStates( profs );
                // それをそのままoutStateManagerにいれる
                std::vector<StatePtr> instates = m_inStateManager.getStates();
                m_outStateManager.addStates(instates);
                m_stateServiceManager.addStates(instates);
                // シーケンサを作って行く
                // シーケンサに必要なのは
                //  上書きすべき値(taeget)
                //  リセットするときに必要な現在値(potentio)
                //  目標位置
                //  目標速度
                //  目標角度
                // の５つ
                // 最後の３つは自分で作って持っておく
                for(SequencerInfoItr it = m_seqInfos.begin(); it != m_seqInfos.end(); it++)
                {
                    // target
                    StatePtr target =  m_inStateManager.getState(it->second.prof.name);
                    // potentio
                    StatePtr potentio =  m_inStateManager.getState(it->second.potentio_name.c_str());
                    // posの作成(targetのtypeが分かるまで作れなかった)
                    RTMEUS::StateType type = target->getStateProfile().type;
                    RTMEUS::StateProfile profile = it->second.prof;
                    profile.type = type;
                    std::string name = it->first + "_pos";
                    profile.name = CORBA::string_dup(name.c_str());
                    StatePtr pos = StateBase::create(profile);
                    m_seqStateManager.addState(pos);
                    // velの作成
                    profile = it->second.prof;
                    name = it->first + "_vel";
                    profile.name = CORBA::string_dup(name.c_str());
                    profile.type = type;
                    StatePtr vel = StateBase::create(profile);
                    m_seqStateManager.addState(vel);
                    // accelの作成
                    profile = it->second.prof;
                    name = it->first + "_accel";
                    profile.name = CORBA::string_dup(name.c_str());
                    profile.type = type;
                    StatePtr accel = StateBase::create(profile);
                    m_seqStateManager.addState(accel);
                    // targettimeの作成
                    RTMEUS::StateProfile tmpprof;
                    name = it->first + "_targettime";
                    tmpprof.name = CORBA::string_dup(name.c_str());
                    tmpprof.type = RTMEUS::DOUBLESEQ_TYPE;
                    tmpprof.length = 1;
                    StatePtr targettime = StateBase::create(tmpprof);
                    m_seqStateManager.addState(targettime);
                    // pasttimeの作成
                    name = it->first + "_pasttime";
                    tmpprof.name = CORBA::string_dup(name.c_str());
                    tmpprof.type = RTMEUS::DOUBLESEQ_TYPE;
                    tmpprof.length = 1;
                    StatePtr pasttime = StateBase::create(tmpprof);
                    m_seqStateManager.addState(pasttime);

                    m_state_target = (float *)target->getPtr(0);
                    m_state_potentio = (float *)potentio->getPtr(0);
                    //priorityのセット
                    for(unsigned int i=0;i<target->getStateProfile().length;i++)
                        current_priorities.push_back(0);
                    //
                    if (m_sequencers.find(it->first) == m_sequencers.end())
                    {
                        SequencerState seqstate;
                        seqstate.sequencer = StateSequencerBase::createStateSequencer(potentio, target, pos, vel, accel);
                        seqstate.sequencer->setSequencerIntegrationmethod(OverRide);
                        m_sequencers.insert(std::pair<std::string, SequencerState>(it->first, seqstate));
                        AnglevectorTimeList avtl;
                        m_AnglevectorTimeLists.insert(std::pair<std::string , AnglevectorTimeList>(it->first, avtl));
                    }
                }
                std::vector<StatePtr> seqstates = m_seqStateManager.getStates();
                m_stateServiceManager.addStates(seqstates);
                //
                m_first = false;
            }
            catch(const std::runtime_error &e)
            {
                std::cerr << e.what() << std::endl;
                return RTC::RTC_ERROR;
            }
        }
    }
    try {
        // 整形
        m_inStateManager <<= m_inState.dataset;

        // 時間の計算
        double dt = m_timer.tick();


        // Sequencerの実行
        // gettimeofdayの時間をつかって次の目標値を計算
        // TODO:Sequencerのタイミングなのか？
        //      要検討
        unsigned int interpolatingfinish_seqnum=0;

        for(SequencerStateItr it = m_sequencers.begin(); it != m_sequencers.end(); it++)
        {
            std::string seqname = it->first;
            it->second.sequencer->stepNext(&current_priorities, dt);

            double ptime = it->second.sequencer->getInterpolatedTime(0);
            memcpy(m_seqStateManager.getState((seqname + "_pasttime").c_str())->getPtr(0), &(ptime), sizeof(double));//targettime

            // 補間がおわっていたらwait-interpolation解除
            if (!it->second.sequencer->hasInterpolator()) {
                it->second.sem->post();
                //priorityのreset
                int seqprio = it->second.sequencer->getSequencerpriority();
                int maskv[(current_priorities.size())];
                it->second.sequencer->getSequencermask(maskv);
                ResetPriorities(maskv, seqprio, (current_priorities.size()));
                interpolatingfinish_seqnum++;
            }
        }
        //線形和の部分のみOverRide後の出力に対して実行
        for(SequencerStateItr it = m_sequencers.begin(); it != m_sequencers.end(); it++)
        {
            int size = m_inStateManager.getState("target")->getLength();
            double sumv[size];

            it->second.sequencer->getSumVector(sumv);
            for(int i=0;i<size;i++) m_state_target[i] += (float)sumv[i];
        }

        // 一つでも補間出力があるならば、その平滑化を行う(がたつき対策）
        if((interpolatingfinish_seqnum!=m_sequencers.size())){
            int size = m_inStateManager.getState("target")->getLength();
            speed_smoothing(m_state_target, size);
        }

        // 整形
        m_outStateManager >>= m_outState.dataset;
    }
    catch(...)
    {
    }
    // 書き込み
    m_outStatePort.write();
    // 条件変数の通知
    scond.broadcast();

    return RTC::RTC_OK;
}

/*
  RTC::ReturnCode_t Seq::onAborting(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/

/*
  RTC::ReturnCode_t Seq::onError(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/

/*
  RTC::ReturnCode_t Seq::onReset(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/

/*
  RTC::ReturnCode_t Seq::onStateUpdate(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/

/*
  RTC::ReturnCode_t Seq::onRateChanged(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/

std::string Seq::angle_vector(const std::string &argv)
{
    if (m_sequencers.size() == 0) return "error: no active sequencer";
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");
    if (argvs.size() == 3) {
        // [0]sequencer-name
        // [1]interpolate-method
        // [2]overwrite
        ScopedLock lock(m_sequencers_lock); // sequencersがクリアされないように
        pop_and_set_angle_vector(argvs[0], argvs[1], (argvs[2] == "t"));
        return ":angle-vector ok";
    } else return "error : invalid argument";
    return ":angle-vector ok";
}

std::string Seq::angle_vector_sequence(const std::string &argv)
{
    if (m_sequencers.size() == 0) return "error: no active sequencer";
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");
    if (argvs.size() == 4) {
        // [0]sequencer-name
        // [1]interpolate-method
        // [2]overwrite
        // [3]length of interpolated sequnces
        ScopedLock lock(m_sequencers_lock); // sequencersがクリアされないように

        unsigned int looptime = atoi(argvs[3].c_str());
        for(unsigned int i=0;i<looptime;i++){
            pop_and_set_angle_vector(argvs[0], argvs[1], (argvs[2] == "t"));
        }
        return ":angle-vector-sequence ok";
    }
    return "error : invalid argument";
}

std::string Seq::pop_and_set_angle_vector(std::string seqname, std::string imethod, bool overwriteflag)
{
    ScopedCondLock(slock, scond);
    
    unsigned long length;
    SequencerStateItr it = m_sequencers.find(seqname); // Seq0
    AnglevectorTimeList *avtl = &(m_AnglevectorTimeLists[seqname]);

    if (it != m_sequencers.end())
    {
        length = m_seqStateManager.getState((seqname + "_pos").c_str())->getStateProfile().length;

        if(avtl->size() > 0){
            std::pair<std::vector<double>, double> avtp;
            RTMEUS::StateType type = m_seqStateManager.getState((seqname + "_pos").c_str())->getStateProfile().type;

            avtp = avtl->front(); //一つ取り出す
            avtl->pop();//削除しておく
            //avtp.firstの方はstateにセット
#define pop_and_set_angle_vector_main(TYPE){                            \
                int siz_ = sizeof(TYPE) * length;                       \
                TYPE avbf[length];                                      \
                TYPE data[length];                                      \
                /*target pos*/                                          \
                std::vector<double> av = avtp.first;                    \
                for(unsigned int i=0;i<length;i++) data[i] = av[i];     \
                if(imethod=="minjerk-sequence") memcpy(avbf, m_stateServiceManager.getState((seqname + "_pos").c_str())->getPtr(0), siz_); \
                memcpy(m_seqStateManager.getState((seqname + "_pos").c_str())->getPtr(0), data,siz_);/*pos*/ \
                /*target vel*/                                          \
                if(imethod=="minjerk-sequence" && avtl->size() > 0){    \
                    std::vector<double> avaf = avtl->front().first;/*次の目標値*/ \
                    double tgt=avtp.second;                             \
                    double tgtaf=avtl->front().second;                  \
                    for(unsigned int i=0;i<length;i++) data[i] = ((av[i] - avbf[i])/ tgt + (avaf[i] - av[i])/ tgtaf)/2.0; \
                } else for(unsigned int i=0;i<length;i++) data[i] = 0.0; \
                memcpy(m_seqStateManager.getState((seqname + "_vel").c_str())->getPtr(0), data,siz_);/*vel*/ \
                /*target accel*/                                        \
                if(imethod=="minjerk-sequence" && avtl->size() > 0){    \
                    std::vector<double> avaf = avtl->front().first;/*次の目標値*/ \
                    double tgt=avtp.second;                             \
                    double tgtaf=avtl->front().second;                  \
                    for(unsigned int i=0;i<length;i++) data[i] = ((av[i] - avbf[i])/ tgt*tgt + (avaf[i] - av[i])/ tgtaf*tgtaf)/2.0; \
                } else for(unsigned int i=0;i<length;i++) data[i] = 0.0; \
                memcpy(m_seqStateManager.getState((seqname + "_accel").c_str())->getPtr(0), data,siz_);/*accel*/ \
            }
            if ( type == RTMEUS::DOUBLESEQ_TYPE ) {
                pop_and_set_angle_vector_main(double);
            }  else if ( type == RTMEUS::FLOATSEQ_TYPE ){
                pop_and_set_angle_vector_main(float);
            } else {
                std::cerr << "UNKNOWN" << std::endl;
            }

            //target time
            memcpy(m_seqStateManager.getState((seqname + "_targettime").c_str())->getPtr(0), &(avtp.second), sizeof(double));//targettime


            if(imethod=="minjerk-sequence") imethod = "minjerk";//methodの上書き
            it->second.sequencer->pushInterpolator(imethod, avtp.second, overwriteflag);
        }        
        return ":angle-vector-sequence ok";
    }
    else {
        return std::string("error: cannot find sequencer :") + seqname;
    }
}

std::string Seq::set_anglevectortimelist(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");
    //argvをparseする
    //[0] sequencer name
    //[1] data length 
    //[2]以後 av00 av01 ... av0n time0 av10 av11 .. av1n time1 ...
    if (argvs.size() > 3) {
        std::string seqname = argvs[0];
        int datapos = 0;
        int datalength = atoi(argvs[1].c_str());
        unsigned long avlength = m_seqStateManager.getState((seqname + "_pos").c_str())->getStateProfile().length;

        for(int i=0;i<datalength;i++){
            std::vector <double> av;
            datapos = i * (avlength + 1)+2;//argvs[2]からなので
            for(unsigned int j=0;j<avlength;j++) av.push_back((double)atof(argvs[datapos++].c_str()));

            std::pair <std::vector<double> , double> avtpair;
            avtpair.first = av;
            avtpair.second = atof(argvs[datapos].c_str())/1000.0;//[ms] -> [s]

            m_AnglevectorTimeLists[seqname].push(avtpair);
        }
        return ":set-anglevectortimelist ok";
    } else return "error : invalid argument";
}

std::string Seq::create_newsequencer(const std::string &argv)
{
    // シーケンサを新たに作る
    //  目標位置
    //  目標速度
    //  目標角度
    //  の３つは各Sequencerごとにもたせる
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    std::string seqname = argvs[0];
    SequencerStateItr ssit = m_sequencers.find(seqname); // Seq0

    if(ssit!=m_sequencers.end()) return ":create-newsequencer : A sequencer with same name exists!";

    SequencerInfoItr it = m_seqInfos.begin();

    // target
    StatePtr target =  m_inStateManager.getState(it->second.prof.name);
    // potentio
    StatePtr potentio =  m_inStateManager.getState(it->second.potentio_name.c_str());
    // posの作成(targetのtypeが分かるまで作れなかった)
    RTMEUS::StateType type = target->getStateProfile().type;
    RTMEUS::StateProfile profile = it->second.prof;
    profile.type = type;
    std::string name = seqname + "_pos";
    profile.name = CORBA::string_dup(name.c_str());
    StatePtr pos = StateBase::create(profile);
    m_seqStateManager.addState(pos);
    // velの作成
    profile = it->second.prof;
    name = seqname + "_vel";
    profile.name = CORBA::string_dup(name.c_str());
    profile.type = type;
    StatePtr vel = StateBase::create(profile);
    m_seqStateManager.addState(vel);
    // accelの作成
    profile = it->second.prof;
    name = seqname + "_accel";
    profile.name = CORBA::string_dup(name.c_str());
    profile.type = type;
    StatePtr accel = StateBase::create(profile);
    m_seqStateManager.addState(accel);
    // targettimeの作成
    RTMEUS::StateProfile tmpprof;
    name = seqname + "_targettime";
    tmpprof.name = CORBA::string_dup(name.c_str());
    tmpprof.type = RTMEUS::DOUBLESEQ_TYPE;
    tmpprof.length = 1;
    StatePtr targettime = StateBase::create(tmpprof);
    m_seqStateManager.addState(targettime);
    // pasttimeの作成
    name = seqname + "_pasttime";
    tmpprof.name = CORBA::string_dup(name.c_str());
    tmpprof.type = RTMEUS::DOUBLESEQ_TYPE;
    tmpprof.length = 1;
    StatePtr pasttime = StateBase::create(tmpprof);
    m_seqStateManager.addState(pasttime);

    //
    SequencerState seqstate;
    seqstate.sequencer = StateSequencerBase::createStateSequencer(potentio, target, pos, vel, accel);
    seqstate.sequencer->setSequencerIntegrationmethod(OverRide);
    m_sequencers.insert(std::pair<std::string, SequencerState>(seqname, seqstate));

    return ":create-newsequencer ok";
}

std::string Seq::set_sequencer_priority(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    SequencerStateItr it = m_sequencers.find(argvs[0]); // Seq0
    int priority=atoi(argvs[1].c_str());

    if (it != m_sequencers.end())
    {
        it->second.sequencer->setSequencerpriority(priority);
    }
    return ":set-sequencer-priority ok";
}

std::string Seq::get_sequencer_priority(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    SequencerStateItr it = m_sequencers.find(argvs[0]); // Seq0
    int priority=0;

    if (it != m_sequencers.end())
    {
        priority = it->second.sequencer->getSequencerpriority();
    }
    std::cerr <<  ":get-sequencer-priority ok" << std::endl;

    char buf[3];
    sprintf(buf,"%d",priority);
    std::string str(buf);

    return str;
}

std::string Seq::set_sequencer_mask(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    SequencerStateItr it = m_sequencers.find(argvs[0]); // Seq0
    int length = m_seqStateManager.getState((argvs[0] + "_pos").c_str())->getStateProfile().length;
    int mask[length];

    for(int i=0;i<length;i++){
        mask[i] = atoi(argvs[i+1].c_str());
    }

    if (it != m_sequencers.end())
    {
        it->second.sequencer->setSequencermask(mask);
    }
    return ":set-sequencer-mask ok";
}

std::string Seq::get_sequencer_mask(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");
    SequencerStateItr it = m_sequencers.find(argvs[0]); // Seq0
    int length = m_seqStateManager.getState((argvs[0] + "_pos").c_str())->getStateProfile().length;
    int mask[length];
    char buf[3+length*2];

    if (it != m_sequencers.end())
    {
        it->second.sequencer->getSequencermask(mask);
    }

    sprintf(&(buf[0]),"#i(");

    for(int i=0;i<length-1;i++) {
        if(mask[i]>0) sprintf(&(buf[3+2*i]),"1 ");
        else sprintf(&(buf[3+2*i]),"0 ");}

    if(mask[length-1]>0) sprintf(&(buf[3+2*(length-1)]),"1");
    else sprintf(&(buf[3+2*(length-1)]),"0");

    sprintf(&(buf[2+length*2]),")");

    std::string str(buf);

    std::cerr << ":get-sequencer-mask ok" << std::endl;

    return str;
}

std::string Seq::set_sequencer_integrationmethod(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    SequencerStateItr it = m_sequencers.find(argvs[0]); // Seq0
    int im=atoi(argvs[1].c_str());

    if (it != m_sequencers.end())
    {
        it->second.sequencer->setSequencerIntegrationmethod(im);
    }
    return ":set-sequencer-integrationmethod ok";
}

std::string Seq::set_sequencer_motionspeed(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    SequencerStateItr it = m_sequencers.find(argvs[0]); // Seq0
    double speed=atof(argvs[1].c_str());

    if (it != m_sequencers.end())
    {
        it->second.sequencer->setSequencerMotionspeed(speed);
    }
    return ":set-sequencer-motionspeed ok";
}

std::string Seq::set_sequencer_motiongain(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    SequencerStateItr it = m_sequencers.find(argvs[0]); // Seq0
    double gain=atof(argvs[1].c_str());

    if (it != m_sequencers.end())
    {
        it->second.sequencer->setSequencerMotiongain(gain);
    }
    return ":set-sequencer-motiongain ok";
}

std::string Seq::wait_interpolation(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    std::string name = argvs[0];
    unsigned long timeout = atol(argvs[1].c_str());

    if (m_sequencers.size() == 0) return "error: no active sequencer";
    {
        //ScopedLock lock(m_sequencers.lock); // 本当はlockをかけたい//TODOで
        SequencerStateItr it = m_sequencers.find(name); // Seq0
        if (it != m_sequencers.end()) {
            // 待ちリストに追加
            it->second.sem->lock();
            if (it->second.sem->wait(timeout) == 0)
            {
                return ":wait-interpolation ok";
            }
            else
            {
                return "error: tiemout";
            }
        }
        return std::string("error: cannot find sequencer :") + argv;
    }
}

std::string Seq::wait_all_interpolation(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    unsigned long timeout = atol(argvs[0].c_str());

    if (m_sequencers.size() == 0) return "error: no active sequencer";
    {
        //ScopedLock lock(m_sequencers.lock); // 本当はlockをかけたい//TODOで
        for(SequencerStateItr it = m_sequencers.begin(); it != m_sequencers.end(); it++)
            if (it != m_sequencers.end()) {
                // 待ちリストに追加
                it->second.sem->lock();
                if (it->second.sem->wait(timeout) != 0) 
                    std::cerr << "warning!: tiemout error" << std::endl;
            }
    }
    return ":wait-interpolation ok";
}

std::string Seq::adapt(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");
    std::string name = argvs[0];

    {
        ScopedLock lock(m_sequencers_lock); // sequencersがクリア去れないように
        if (m_sequencers.size() == 0) return "error: no active sequencer";

        SequencerStateItr it = m_sequencers.find(name); // Seq0
        if (it != m_sequencers.end()) {
            it->second.sequencer->adapt();
            return ":adapt ok";
        }
    }
    return std::string("error: cannot find sequencer : ") + argv;
}

std::string Seq::adapt_1motor(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");
    std::string name = argvs[0];
    int motorno = atoi(argvs[1].c_str());

    if (argvs.size() == 1) {
        ScopedLock lock(m_sequencers_lock); // sequencersがクリア去れないように
        SequencerStateItr it = m_sequencers.find(name); // Seq0

        if (it != m_sequencers.end()) {
            it->second.sequencer->adapt_1motor(motorno);
            return ":adapt-1motor ok";
        }
    }
    return "error : invalid argument";
}

std::string Seq::adapt_allseq(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    ScopedLock lock(m_sequencers_lock); // sequencersがクリア去れないように
    if (m_sequencers.size() == 0) return "error: no active sequencer";

    for(SequencerStateItr it = m_sequencers.begin(); it != m_sequencers.end(); it++)
        it->second.sequencer->adapt();
    return ":adapt_allseq ok";
}

std::string Seq::adapt_1motor_allseq(const std::string &argv)
{
    std::vector<std::string> argvs;
    argvs = coil::split(argv ," ");

    int motorno = atoi(argvs[0].c_str());

    if (argvs.size() == 1) {
        ScopedLock lock(m_sequencers_lock); // sequencersがクリア去れないように

        for(SequencerStateItr it = m_sequencers.begin(); it != m_sequencers.end(); it++)
            it->second.sequencer->adapt_1motor(motorno);
        return ":adapt-1motor ok";
    } else return "error : invalid argument";
}

//ただの速度制限
void Seq::speed_smoothing(float* curv, int size)
{
    float diff;

    for(int i=0;i<size;i++){
        diff = curv[i] - m_state_potentio[i];
        if(fabs(diff) > m_anglevel_limit){
            std::cerr << "AngleVel Limited" << std::endl;
            if(diff > 0) curv[i] = m_state_potentio[i] +  m_anglevel_limit;
            else curv[i] = m_state_potentio[i] - m_anglevel_limit;
        }
    }
}

extern "C"
{

    void SeqInit(RTC::Manager* manager)
    {
        coil::Properties profile(seq_spec);
        manager->registerFactory(profile,
                                 RTC::Create<Seq>,
                                 RTC::Delete<Seq>);
    }

};


