// -*- C++ -*-
#ifndef __SEQUENCER_H__
#define __SEQUENCER_H__

#include <string>
#include <list>
#include <iostream>
#include <utils/Mutex.h>
#include <boost/utility.hpp>
#include "Interpolator.h"

// Sequencerするクラス
// mutexをかけているのでnoncopyable(コピーできないクラス)
// コピー的なことをするとエラー
template<typename V, typename T = double>
class Sequencer
    : public boost::noncopyable // ポインタを持っているのでコピー不可に
{
public:
    Sequencer() : m_time(0), m_x(0), m_vel(0), m_accel(0) {}
    virtual ~Sequencer() {
        clearAll();
    }

    V getPos() { return m_x; }
    V getVel() { return m_vel; }
    V getAccel() { return m_accel; }

    bool hasInterpolator() {return !m_interpolator_list.empty();}
    void setPos(V x) { m_x = x; clearAll();}

    // dtだけ進める
    bool stepNext(T dt)
        {
            if (m_interpolator_list.empty()) return false; // 何もなかったら何もしない
            {
                ScopedLock lock(m_mutex);
                typename std::list< Interpolator<V,T>*  >::iterator it = m_interpolator_list.begin();
                m_time += dt;
                if (m_time > (*it)->getTime()) {
                    m_time = m_time - (*it)->getTime(); // ちょっと過ぎてるかもしれないので0にはしない
                    if (*it == m_interpolator_list.back()) {
                        m_x = (*it)->getFinalPos();
                        m_vel = m_accel = 0;
                    }
                    delete (*it);
                    it = m_interpolator_list.erase(it);
                    if (m_interpolator_list.empty()) 
                    {
                        return true;
                    }
                }
                m_x = (*it)->getPos(m_time);
                m_vel = (*it)->getVel(m_time);
                m_accel = (*it)->getAccel(m_time);
            }
            return true;
        }
    // 一個push
    void pushInterpolator(std::string name, T time, V xf, V vf = 0, V af = 0) {
        V x,v,a;
        // 終わりの状態を取ってくる
        if (m_interpolator_list.empty()) {
            x = m_x; v = m_vel; a = m_accel;
        }
        else {
            Interpolator<V,T>* last = m_interpolator_list.back();
            x = last->getFinalPos(); v = last->getFinalVel(); a = last->getFinalAccel();
        }
        // Interpolatorを作る
        Interpolator<V,T> *new_in;
        if (name == "minjerksequence") {
            if (!(new_in = Interpolator<V,T>::create("minjerk"))) return;
            // 境界の速度，加速度を直す
            v = (vf + v) / 2;
            a = (af + a) / 2;
            recalcLast(x,v,a);
        }
        else {
            if (!(new_in = Interpolator<V,T>::create(name))) return;
        }
        // 境界条件設定
        new_in->setBoundaryCondition(time, x, v, a, xf, vf, af);
        {
            ScopedLock lock(m_mutex);
            m_interpolator_list.push_back(new_in);
        }
    }
    // clear
    void clearAll(bool reset = true)
        {
            typename std::list< Interpolator<V,T>*  >::iterator it;
            {
                ScopedLock lock(m_mutex);
                for(it = m_interpolator_list.begin(); it != m_interpolator_list.end();
                    it = m_interpolator_list.erase(it))
                    delete (*it);
                if (reset)
                {
                    m_vel = m_accel = 0;
                }
                m_time = 0;
            }
        }
    // m_timeを返す
    double getInterpolatedTime()
        {
            return m_time;
        }

protected:
    T m_time;
    V m_x, m_vel, m_accel;
    coil::Mutex m_mutex;
    std::list< Interpolator<V,T>* > m_interpolator_list;
    // 一番最後のInterpolatorの終わりの境界条件を変更する
    void recalcLast(V xf, V vf, V af)
        {
            if (m_interpolator_list.empty()) return; // 何もなかったら何もしない
            {
                ScopedLock lock(m_mutex);
                Interpolator<V,T>* last = m_interpolator_list.back();
                T time = last->getTime();
                if ((last = m_interpolator_list.front())!=NULL) { // 補間中なら
                    last->setBoundaryCondition(time - m_time, m_x, m_vel, m_accel, xf, vf, af);
                    m_time = 0;
                }
                else {
                    last->setBoundaryCondition(m_time, xf, vf, af);
                }
            }
        }
};
 
#endif // __SEQUENCER_H__
