// -*- C++ -*-
#include <boost/shared_ptr.hpp>
#include <vector>
#include <services/StateService/State.h>
#include "Sequencer.h"

#ifndef __STATESEQUENCER_H__
#define __STATESEQUENCER_H__

typedef enum {OverRide, LinearSum} IntegrationMethod;
	
class StateSequencerBase;
typedef boost::shared_ptr<StateSequencerBase> StateSequencerPtr;

// interfaceとfactory
// 複数のSequencerUnitをもつ
class StateSequencerBase
{
public:
    // factory
    virtual ~StateSequencerBase() {}
    static StateSequencerPtr createStateSequencer(const StatePtr &potentio,
                                                  const StatePtr &target,
                                                  const StatePtr &reference_pos,
                                                  const StatePtr &reference_vel,
                                                  const StatePtr &reference_accel);
    // interfaces
    virtual void adapt() = 0;
    virtual void adapt_1motor(int i) = 0;
    virtual void stepNext(std::vector<int> *curprios, double t) = 0;
    virtual void pushInterpolator(const std::string &name, double time, bool overwrite=false) = 0;
    virtual bool hasInterpolator() = 0;
		virtual void setSequencerpriority(int priority) = 0;
		virtual int getSequencerpriority(void) = 0;
    virtual void setSequencermask(int *maskv) = 0;
    virtual void getSequencermask(int *maskv) = 0;
    virtual double getInterpolatedTime(int seqno) = 0;
		virtual void getSumVector(double *sumv) = 0;
		virtual void setSequencerIntegrationmethod(int im) = 0;
		virtual void setSequencerMotionspeed(double speed) = 0;
		virtual void setSequencerMotiongain(double gain) = 0;
};

// 実装クラス
template<typename SRC, typename DST, typename T = double>
class StateSequencer
    : public StateSequencerBase
{
public:
    // destructor
    virtual ~StateSequencer() {};
    //constructor
    StateSequencer(const StatePtr &potentio,
                   const StatePtr &target,
                   const StatePtr &reference_pos,
                   const StatePtr &reference_vel,
                   const StatePtr &reference_accel)
        // 自分も参照を保持しておく(念のため)
        : m_potentio(potentio),
          m_target(target),
          m_reference_pos(reference_pos),
          m_reference_vel(reference_vel),
          m_reference_accel(reference_accel)
        {
            // profにしたがって作る
            const RTMEUS::StateProfile &profs = reference_pos->getStateProfile();
            for(unsigned int i = 0; i < profs.length; i++)
            {
                m_sequencers.push_back(
                    SequencerUnit((SRC*)potentio->getPtr(i),
                                  (DST*)target->getPtr(i),
                                  (DST*)reference_pos->getPtr(i),
                                  (DST*)reference_vel->getPtr(i),
                                  (DST*)reference_accel->getPtr(i)));
            }
            //maskを作る
            for(unsigned int i = 0; i < profs.length; i++) m_sequencermask.push_back(1);
            //priorityのセット
            m_priority = 0;
            //motionspeed, motiongain
            m_motionspeed = 1.0;
            m_motiongain = 1.0;
            //sumvectorを作る
            for(unsigned int i = 0; i < profs.length; i++) m_sumvector.push_back((DST)0.0);
            // default imethod
            m_imethod = OverRide;
        }
    // 次の値をセットする
    virtual void stepNext(std::vector<int> *curprios, double t)
        {
            SequencerUnit *s = &m_sequencers[0];
            for(unsigned int i = 0; i < m_sequencers.size(); i++) 
            {
                bool IsInterpolated;
                IsInterpolated = s->sequencer->stepNext((T)t);

                switch(m_imethod){
                case OverRide:
                    if((m_sequencermask[i] > 0) && ((*curprios)[i] <= m_priority)){
                        *(s->dst) = (DST)s->sequencer->getPos();
                        (*curprios)[i] = m_priority;
                    }
                    m_sumvector[i] = (DST)0.0;
                    break;
                case LinearSum:
                    m_sumvector[i] = (DST)(m_sequencermask[i] * s->sequencer->getPos());
                    break;
                default:
                    std::cerr << "Unsupported Integration Method!" << std::endl;
                    break;
                }
                s++;
            }
				
        }
    // 次の目標値をセットする
    virtual void pushInterpolator(const std::string &name, double time, bool overwrite=false)
        {
            // 始めてなら強制adapt(Only OverRide mode)
            if (!hasInterpolator()) 
            {
                if (m_imethod!=LinearSum) adapt();
            }
            SequencerUnit *s = &m_sequencers[0];
            for(unsigned int i = m_sequencers.size(); i > 0; i--) 
            {
                if (overwrite) s->sequencer->clearAll(false); 					
                s->sequencer->pushInterpolator(name, (time*m_motionspeed), (SRC)((*s->pos)*m_motiongain), (SRC)((*s->vel)*m_motiongain), (SRC)((*s->accel)*m_motiongain));
                s++;
            }
        }
    virtual bool hasInterpolator() 
        {
            if (m_sequencers.size() == 0) return false;
            return m_sequencers[0].sequencer->hasInterpolator();
        }
    virtual void adapt() 
        {
            SequencerUnit *s = &m_sequencers[0];
            if(m_imethod!=LinearSum){
                for(unsigned int i = m_sequencers.size(); i > 0; i--) 
                {
                    s->sequencer->setPos((DST)*(s->src));
                    s++;
                }
            }
        }
    virtual void adapt_1motor(int i) 
        {
            if((i < 0) || (i >= (int)m_sequencers.size())) {
                std::cerr << "Invalid arguments in adapt_1motor" << std::endl;
                return;}
            SequencerUnit *s = &m_sequencers[i];
            if(m_imethod!=LinearSum)
                s->sequencer->setPos((DST)*(s->src));
        }
    // priorityをセットする
    virtual void setSequencerpriority(int priority)
        {
            m_priority = priority;
        }
    // 現在のpriorityを返す
    virtual int getSequencerpriority(void)
        {
            return m_priority; 
        }
    // マスクをセットする
    virtual void setSequencermask(int *maskv)
        {
            for(unsigned int i = 0; i < m_sequencers.size(); i++)  m_sequencermask[i] = maskv[i];
        }
    // 現在のマスクを返す
    virtual void getSequencermask(int *maskv)
        {
            for(unsigned int i = 0; i < m_sequencers.size(); i++)  maskv[i] = m_sequencermask[i];
        }
    // 補間の残り時間を得る
		virtual double getInterpolatedTime(int seqno)
        {
            SequencerUnit *s = &m_sequencers[0]; 
            return s->sequencer->getInterpolatedTime();
        }
    // 補間の残り時間を得る
		virtual void getSumVector(double *sumv)
        {
            for(unsigned int i = 0; i < m_sequencers.size(); i++)  sumv[i] = (double)m_sumvector[i];
        }
		//
		virtual void setSequencerIntegrationmethod(int im)
        {
            switch(im){
            case OverRide:
                m_imethod = OverRide;
                break;
            case LinearSum:
                m_imethod = LinearSum;
                break;
            default:
                std::cerr << "Unsupported Integration Method!" << std::endl;
                break;
            }
        }
		//
		virtual void setSequencerMotionspeed(double speed)
        {
            m_motionspeed = speed;
        }
		//
		virtual void setSequencerMotiongain(double gain)
        {
            m_motiongain = gain;
        }
		//
private:
    // 参照
    StatePtr m_potentio, m_target, m_reference_pos, m_reference_vel,m_reference_accel;
    
    // シーケンサとポインタ管理するクラス
    typedef boost::shared_ptr< Sequencer<SRC, T> > SeqPtr;

    struct SequencerUnit {
        SequencerUnit(SRC *s, DST *d, DST *p, DST* v, DST* a)
            : src(s), dst(d), pos(p), vel(v), accel(a)
            {
                sequencer = SeqPtr(new Sequencer<SRC, T>());
            }
        SeqPtr sequencer;
        //
        SRC *src;
        DST *dst;
        //
        DST *pos;
        DST *vel;
        DST *accel;
    private:
    };
    std::vector< SequencerUnit > m_sequencers; 
    std::vector < int > m_sequencermask;
		std::vector < DST > m_sumvector;
    int m_priority;
		double m_motionspeed;
		double m_motiongain;
		IntegrationMethod m_imethod;
};

inline StateSequencerPtr StateSequencerBase::createStateSequencer(const StatePtr &potentio,
                                                                  const StatePtr &target,
                                                                  const StatePtr &reference_pos,
                                                                  const StatePtr &reference_vel,
                                                                  const StatePtr &reference_accel)
{
    RTMEUS::StateType src_type = potentio->getStateProfile().type;
    RTMEUS::StateType dst_type = target->getStateProfile().type;
    if (src_type == RTMEUS::DOUBLESEQ_TYPE && dst_type == RTMEUS::DOUBLESEQ_TYPE)
        return StateSequencerPtr
            (new StateSequencer<double, double>(potentio,
                                                target,
                                                reference_pos, 
                                                reference_vel,
                                                reference_accel));
    if (src_type == RTMEUS::FLOATSEQ_TYPE && dst_type == RTMEUS::FLOATSEQ_TYPE)
        return StateSequencerPtr
            (new StateSequencer<float, float>(potentio,
                                              target,
                                              reference_pos,
                                              reference_vel,
                                              reference_accel));
    throw EInvalidStateType(" Sequencer inout");
}

#endif 
// end of file
